/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xci.c
 *       Completer-I9nitiator (CI) functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xregcons.h>
#include <xsession.h>
#include <xci.h>

#include <xsetting.h>
#include <xaddrmap.h>
#include <xloader.h>

#include <xregcons.h>
#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>

#if 0 /* HL, not needed ??? */
#include <xpci.h>
#include <timeout.h>
#endif

/* Pointer to completer database */
#define DB_COMPLETER (bx_handlearray[handle].db->Exerciser.CompleterInitiator)

/* Value of generic property */
#define GENERIC(prop) (DB_COMPLETER.GenProp[prop])

/* Pointer to one line in behavior-memory */
#define BEHAVIOR(line) (DB_COMPLETER.Behavior.Mem[line])

/* Pointer to Behavior Update Info */
#define BEH_UPD (&DB_COMPLETER.Behavior.UpdateInfo)

/* memory map moved to xdyndata.c */

/* the following lines leaves running code running.... */
extern bx_memmaptype Mephisto_CI_Beh2Hw [];
#define Beh2Hw Mephisto_CI_Beh2Hw

/*****************************************************************************
 * CI Programming
 ****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXCIProg
 *
 * Purpose: Program properties and memories from host to card
 *---------------------------------------------------------------------------*/
bx_errtype BestXCIProg (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXCIProg [ciprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXCIBehProg(handle));
    BX_TRY(BestXCIGenProg(handle));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* Prog functions successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* Prog() functions not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIRead
 *
 * Purpose: Read properties and memories from card- to host memory 
 *---------------------------------------------------------------------------*/
bx_errtype BestXCIRead (bx_handletype handle,bx_int32 option)
{
  BX_DECLARE_FUNCNAME("BestXCIRead [ciread]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXCIBehRead(handle,option));
    BX_TRY(BestXCIGenRead(handle));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/****************************************************************************
   CI generics **************************************************************
 ****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXCIGenGet 
 *
 * Purpose: Gets a generic property from host-memory 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIGenGet(
  bx_handletype handle,
  bx_cigentype prop,
  bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXCIGenGet [cigenget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_CI_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    switch (prop)
    {
      case BX_CIGEN_NUMBEH:

        /* BX_CIGEN_NUMBEH does not exist in HW.
           If user gets this property, he
           implicitly gets it from Group 0
           (all groups usually have same value):
        */
        BX_TRY(BestXCIGenGet(handle, BX_CIGEN_NUMBEHG0, val));
        break;

      default:
        /* get property from host DB */
        *val=(bx_int32)GENERIC(prop);
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIGenSet         
 *
 * Purpose: Sets a generic property (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIGenSet(
  bx_handletype handle,
  bx_cigentype prop,
  bx_int32 val)

{
  BX_DECLARE_FUNCNAME("BestXCIGenSet [cigenset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_CI_GEN, (bx_int32)prop, val));

    switch (prop)
    {
      case BX_CIGEN_NUMBEH:

        /* BX_CIGEN_NUMBEH does not exist in HW.
           If user sets this property, he
           implicitly sets all groups:
        */
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG0, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG1, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG2, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG3, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG4, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG5, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG6, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG7, val));
        BX_TRY(BestXCIGenSet(handle, BX_CIGEN_NUMBEHG8, val));
        break;

      default:
        /* set property in host DB */
        GENERIC(prop)=val;
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIGenDefaultSet
 *
 * Purpose: Set all generic properties to default (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIGenDefaultSet(
bx_handletype handle)

{
  BX_DECLARE_FUNCNAME("BestXCIGenDefaultSet [cigendefset]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_CI_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_CI_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXCIGenSet(handle,
              ParamInfo->proptyp.cigenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXCIGenProg
 *
 * Purpose: Writes all generic properties to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCIGenProg(bx_handletype  handle)
{
  /* Updates Generics memory on card */

  BX_DECLARE_FUNCNAME("BestXCIGenProg [cigenprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0, regval=0;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXCIGenGet (handle,  BX_CIGEN_NUMBEHG0, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle,  BX_CIGEN_NUMBEHG1, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle,  BX_CIGEN_NUMBEHG2, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle,  BX_CIGEN_NUMBEHG3, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle,  BX_CIGEN_NUMBEHG4, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_NUMBEHG5, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR5_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_NUMBEHG6, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR6_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_NUMBEHG7, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR7_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_NUMBEHG8, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_CTR8_LOOP_REG, sizeof(bx_int16), value-1));

    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_MESSAGE_AD, &value)); 
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CSPLIT_MESSAGE_REG, sizeof(bx_int32), value));
    

    /* the following properties are all coded in one WORD register. */
    /* see address_map for bit positions! */
    regval = 0UL;
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ATTR28, &value));
    if (value)
    {
      regval |= 1UL << 7;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ATTR27, &value));
    if (value)
    {
      regval |= 1UL << 6;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ATTR26, &value));
    if (value)
    {
      regval |= 1UL << 5;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ATTR25, &value));
    if (value)
    {
      regval |= 1UL << 4;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ATTR24, &value));
    if (value)
    {
      regval |= 1UL << 3;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ADDR31, &value));
    if (value)
    {
      regval |= 1UL << 2;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ADDR30, &value));
    if (value)
    {
      regval |= 1UL << 1;
    }
    BX_TRY(BestXCIGenGet (handle, BX_CIGEN_ADDR7, &value));
    if (value)
    {
      regval |= 1UL;
    }

    BX_TRY(BestXDirectRegWrite(handle, BX_REG_CBEHAV_RSVD_REG, sizeof(bx_int16), regval));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIGenRead
 *
 * Purpose: Reads all generic properties from card- to host-memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCIGenRead(bx_handletype  handle)
{
  /* Updates generics memory on host DB */

  BX_DECLARE_FUNCNAME("BestXCIGenRead [cigenread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0, regval=0;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEHG0, value+1));
    
    /* Arbitrary: take value from group 0 */
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEH, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEHG1, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEHG2, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEHG3, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle,  BX_CIGEN_NUMBEHG4, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR5_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_NUMBEHG5, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR6_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_NUMBEHG6, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR7_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_NUMBEHG7, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR8_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_NUMBEHG8, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_CSPLIT_MESSAGE_REG, sizeof(bx_int32), &value));
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_MESSAGE_AD, value)); 


    /* the following properties are all coded in one WORD register. */
    /* see address_map for bit positions! */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_RSVD_REG, sizeof(bx_int16), &regval));
      
    value = (regval & (1UL << 7)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ATTR28, value));
    value = (regval & (1UL << 6)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ATTR27, value));
    value = (regval & (1UL << 5)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ATTR26, value));
    value = (regval & (1UL << 4)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ATTR25, value));
    value = (regval & (1UL << 3)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ATTR24, value));
    value = (regval & (1UL << 2)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ADDR31, value));
    value = (regval & (1UL << 1)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ADDR30, value));
    value = (regval & (1UL << 0)) ? 1 : 0;
    BX_TRY(BestXCIGenSet (handle, BX_CIGEN_ADDR7, value));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/****************************************************************************
   CI behavior **************************************************************
 ****************************************************************************/
/*---------------------------------------------------------------------------*
 * BestXCIBehMemInit
 *
 * Purpose: Defaults the complete behavior memory (on host only)
 * Caution: A following ExerciserProg() will re-program whole memory !!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehMemInit(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXCIBehMemInit [cibehmeminit]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 offset;
  
  BX_TRY_BEGIN
  {
    for (offset=0;offset<BX_CIBEH_MEMDEPTH;offset++)
    {
      BX_TRY(BestXCIBehDefaultSet(handle,offset));
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehDefaultSet
 *
 * Purpose: Defaults one behavior line (on host)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehDefaultSet(
  bx_handletype handle,
  bx_int32 offset)
{
  BX_DECLARE_FUNCNAME("BestXCIBehDefaultSet []");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_CI_BEH, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_CI_BEH,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXCIBehSet(handle,offset,
              ParamInfo->proptyp.cibehprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehGet
 *
 * Purpose: Gets a behavior property (from host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehGet(
  bx_handletype handle,
  bx_int32 offset,
  bx_cibehtype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME("BestXCIBehGet [cibehget]");

  bx_int32 i;
  bx_int32 ival;  /* partial bits of property value */

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Get pointer to i-th (existing) property */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_CI_BEH, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */
    
    *val=0;
    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we get corresponding bits */       
        ival=0;
        if (!BestXMemPropGet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16),  /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,                     /* first bitpos to read */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to read */
          &ival,                                      /* returned bits */
          Beh2Hw[i].PropName,                         /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }

        /* merge read bits of ival into *val */
        *val = *val | ival<<Beh2Hw[i].PropBitPos; /* set bits */
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }
  
  /* We need some modifications to adapt sw to hw */
  switch (prop)    
  {
    case BX_CIBEH_REPEAT:
      if (*val==0)
      {
        /* This register is 8 bit wide in HW.
           A value of zero in HW means 256 in CAPI.
        */
        *val=0x100UL;
      }
      break;
    
    default:
      break;
  }
  
  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehSet
 *
 * Purpose: Sets a behavior property (host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehSet(
  bx_handletype handle,
  bx_int32 offset,
  bx_cibehtype prop,
  bx_int32 val
)

{
  BX_DECLARE_FUNCNAME("BestXCIBehSet [cibehset]");
  bx_int32 i;

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* dynamic capability checking concerning values of parameters */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_CI_BEH, (bx_int32)prop, val));

    /* We need some modifications to adapt sw to hw */
    switch (prop)        
    {
      case BX_CIBEH_REPEAT:
        if (val==0x100UL)
        {
          /* This register is 8 bit wide in HW.
             A value of zero in HW means 256 in CAPI
          */
      val=0UL;
        }
        break;
            
      default:
        break;
    }

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */

    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we set corresponding bits */       
 
        if (BestXMemPropSet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16), /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,                     /* first bitpos to change */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to change */
          val>>Beh2Hw[i].PropBitPos, /* relevant bits to set start at position 0 */
          Beh2Hw[i].PropName,        /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          /* success */
          /* mark this property modified */
          SetUpdateInfo(BEH_UPD,1,offset,Beh2Hw[i].MemBitPos/16);  
        }
        else
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehProg
 *
 * Purpose: Writes changes of behavior memory from host to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehProg(bx_handletype  handle)
{
  /* Updates completer behavior on card */

  BX_DECLARE_FUNCNAME("BestXCIBehProg [cibehprog]");

  BX_TRY_VARS;

  bx_bool  Touched;  /* Wether memory needs an update at all */
  bx_int32 FirstRow,LastRow,FirstCol,LastCol; /* Range to update */

  bx_int32 i,row,col;  /* Indices */

  bx_int16ptr data=NULL;    /* part of behavior memory to be written to card */
  bx_int32 NumWords;   /* size of data[] in words */

  /* Retrieve info about the range to update */
  GetUpdateInfo(BEH_UPD,&Touched,&FirstRow,&LastRow,&FirstCol,&LastCol);
  
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 
    if (!Touched)
    {
      /* host and card memory are identical, no update needed */
    }
    else
    {
      /* host- and card memory differ */
    
      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* Min and Max registers determine the Column (property) -range 
         to program 
      */
      assert(FirstCol<BX_CIBEH_MEMWIDTH);
      assert(LastCol<BX_CIBEH_MEMWIDTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_MIN_REG,sizeof(bx_int16),FirstCol));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_MAX_REG,sizeof(bx_int16),LastCol));

      /* Set starting row (i.e. offset) from which to start programming */
      assert(FirstRow<BX_CIBEH_MEMDEPTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_CTR,sizeof(bx_int16),FirstRow));

      /* In order to use a single call to BestXDirectRegBlockWrite() 
         instead of many calls to BestXDirectRegWrite(),
         we first have to copy the needed behavior properties into 
         a local data array 
      */
      
      NumWords=(LastRow-FirstRow+1) * (LastCol-FirstCol+1);
      data=(bx_int16ptr) BestXMemCalloc(NumWords,sizeof(bx_int16));
      if (data==NULL)
      {
        BX_TRY_FAIL(BX_E_HOST_MEM_FULL);
      }

      /* Fill data[] with needed properties */
      i=0;
      for (row=FirstRow;row<=LastRow;row++)
      {
        for (col=FirstCol;col<=LastCol;col++)
        {
          data[i++]=DB_COMPLETER.Behavior.Mem[row][col];
        }
      }

      assert(i==NumWords);

      /* Write to target behavior memory */
      BX_TRY(BestXDirectRegBlockWrite(handle,
                                      BX_REG_CBEHAV_DATA_REG,
                      sizeof(bx_int16),
                                      1, /* 1=autoincrement */
                                      (bx_int8ptr) data,
                                      sizeof(bx_int16),
                                      NumWords*sizeof(bx_int16)
                                      ));

      BestXMemFree((void**) &data);

      /* Mark host memory to be identical with HW */
      SetUpdateInfo(BEH_UPD,0,0,0);

      /* Switch to Run-Mode (PCI clock) */
      BX_TRY_PROGRESS(BX_E_OK);
      BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
    }
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (data)
        {
          BestXMemFree((void**) &data);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehRead
 *
 * Purpose: Reads behavior memory from card to host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXCIBehRead(bx_handletype  handle,bx_int32 option)
{
  /* Updates behavior memory on host DB */

  BX_DECLARE_FUNCNAME("BestXCIBehRead [cibehread]");

  BX_TRY_VARS;

  bx_int32 NumLines=0;   /* number of lines to read */

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* Min and Max registers determine the Column (property) -range 
       to read from
    */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_MIN_REG,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_MAX_REG,sizeof(bx_int16),BX_CIBEH_MEMWIDTH-1));

    /* Set starting row (i.e. offset) from which to start reading */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_CBEHAV_CTR,sizeof(bx_int16),0));

    /* Determine number of lines to read */
    switch (option)
    {
      case BX_EXEREAD_CARD:
        /* 
           The number of lines to read is determined 
           by current HW values.
        */    
        BX_TRY(BestXDirectRegRead(handle, BX_REG_CBEHAV_CTR0_LOOP_REG,sizeof(bx_int16),&NumLines));
        NumLines++; /* HW value is off by one */
        break;
      case BX_EXEREAD_ALL:
        /* Read complete memory */
        NumLines=BX_CIBEH_MEMDEPTH;
        break;
      case BX_EXEREAD_NONE:
        /* Read nothing */
        NumLines=0;
        break;
      case BX_EXEREAD_USER:
        /* 
           User can determine how many lines are read from card 
           by setting corresponding generic property before.
           Remark:
           During BestXExerciserRead(), the ..GenRead() 
           function must be called after this function,
           because it would overwrites the property with HW-values !!!
         */
        BX_TRY(BestXCIGenGet(handle, BX_CIGEN_NUMBEH, &NumLines));
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXExerciserRead: Invalid value for parameter option");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }

    /* Read behavior memory from card */
    BX_TRY(BestXDirectRegBlockRead(handle,
                                    BX_REG_CBEHAV_DATA_REG,
                    sizeof(bx_int16),
                                    1, /* 1=autoincrement */
                                    (bx_int8ptr)DB_COMPLETER.Behavior.Mem,
                                    sizeof(bx_int16),
                                    NumLines*BX_CIBEH_MEMWIDTH*sizeof(bx_int16)
                                    ));



    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  /* chris: 
     Do not change the update info here, because BestXOpen()
     marks the whole memory valid (except first line).
     SetUpdateInfo(BEH_UPD,0,0,0);
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXCIBehCopyLine
 *
 * Purpose: Copies one behavior line to another.
 *          Used only by application.
 *          Do not document.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCIBehCopyLine(
  bx_handletype  handle,
  bx_int32 srcline,
  bx_int32 destline
)
{
  BX_DECLARE_FUNCNAME("BestXCIBehCopyLine [cibehcopyline]");
  
  bx_int32 col;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    assert(srcline<BX_CIBEH_MEMDEPTH);    
    assert(destline<BX_CIBEH_MEMDEPTH);    

    /* Set the destination line to default, i.e.
       touch it, so that it will be programmed to card
       by ExerciserProg().
    */
    BX_TRY(BestXCIBehDefaultSet(handle,destline));

    /* Copy line directly in DB */
    for (col=0;col<BX_CIBEH_MEMWIDTH;col++)
    {
      BEHAVIOR(destline)[col]=BEHAVIOR(srcline)[col];
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}
